"""
Enhanced Audit Logger for AILinux Backend
Generated by Mistral Large, reviewed by Claude

Features:
- Structured JSON logs with rotation
- Multiple log levels including SECURITY
- SIEM export support (JSON-Lines)
- Retention policy (30 days, 100MB max)
- Async-capable
"""

import json
import logging
import time
import asyncio
from datetime import datetime, timedelta
from logging.handlers import RotatingFileHandler
from pathlib import Path
from typing import Any, Dict, List, Optional
from uuid import uuid4
import gzip
import shutil

from pydantic import BaseModel, Field


class AuditLogEntry(BaseModel):
    """Structured audit log entry."""
    timestamp: str = Field(default_factory=lambda: datetime.utcnow().isoformat() + "Z")
    trace_id: str = Field(default_factory=lambda: str(uuid4())[:12])
    session_id: Optional[str] = None
    user_id: Optional[str] = None
    action: str
    resource: str
    details: Dict[str, Any] = Field(default_factory=dict)
    ip_address: Optional[str] = None
    duration_ms: Optional[float] = None
    log_level: str = "INFO"
    service: str = "ailinux-backend"
    version: str = "2.60"


class EnhancedAuditLogger:
    """
    Enhanced audit logger with structured logging, rotation, and SIEM export.

    Usage:
        audit = EnhancedAuditLogger()
        await audit.log("INFO", "user_login", "/auth/login", {"user": "admin"})
    """

    # Custom log level for security events
    SECURITY = 60
    logging.addLevelName(SECURITY, "SECURITY")

    def __init__(
        self,
        log_dir: Optional[str] = None,
        max_bytes: int = 100 * 1024 * 1024,  # 100MB
        backup_count: int = 10,
        retention_days: int = 30
    ):
        # Default: ./triforce/logs/audit
        if log_dir is None:
            self.log_dir = Path(__file__).parent.parent.parent / "triforce" / "logs" / "audit"
        else:
            self.log_dir = Path(log_dir)
        self.log_dir.mkdir(parents=True, exist_ok=True)
        self.max_bytes = max_bytes
        self.backup_count = backup_count
        self.retention_days = retention_days

        self.logger = self._setup_logger()
        self._buffer: List[AuditLogEntry] = []
        self._buffer_lock = asyncio.Lock()
        self._last_cleanup = time.time()

    def _setup_logger(self) -> logging.Logger:
        """Setup rotating file logger."""
        logger = logging.getLogger("ailinux.audit.enhanced")
        logger.setLevel(logging.DEBUG)

        # Remove existing handlers
        logger.handlers.clear()

        # JSON file handler with rotation
        file_handler = RotatingFileHandler(
            self.log_dir / "audit.jsonl",
            maxBytes=self.max_bytes,
            backupCount=self.backup_count,
            encoding="utf-8"
        )
        file_handler.setFormatter(logging.Formatter('%(message)s'))
        file_handler.setLevel(logging.DEBUG)
        logger.addHandler(file_handler)

        # Security-only file
        security_handler = RotatingFileHandler(
            self.log_dir / "security.jsonl",
            maxBytes=self.max_bytes // 10,
            backupCount=5,
            encoding="utf-8"
        )
        security_handler.setFormatter(logging.Formatter('%(message)s'))
        security_handler.setLevel(self.SECURITY)
        logger.addHandler(security_handler)

        return logger

    def _get_log_level(self, level: str) -> int:
        """Convert string level to logging level."""
        level_map = {
            "DEBUG": logging.DEBUG,
            "INFO": logging.INFO,
            "WARNING": logging.WARNING,
            "ERROR": logging.ERROR,
            "CRITICAL": logging.CRITICAL,
            "SECURITY": self.SECURITY,
        }
        return level_map.get(level.upper(), logging.INFO)

    async def log(
        self,
        level: str,
        action: str,
        resource: str,
        details: Dict[str, Any] = None,
        user_id: Optional[str] = None,
        session_id: Optional[str] = None,
        trace_id: Optional[str] = None,
        ip_address: Optional[str] = None,
        duration_ms: Optional[float] = None
    ) -> AuditLogEntry:
        """
        Log an audit entry.

        Args:
            level: Log level (DEBUG, INFO, WARNING, ERROR, CRITICAL, SECURITY)
            action: Action being performed (e.g., "user_login", "llm_call")
            resource: Resource being accessed (e.g., "/v1/chat/completions")
            details: Additional details dict
            user_id: User identifier
            session_id: Session identifier
            trace_id: Trace ID for request correlation
            ip_address: Client IP address
            duration_ms: Operation duration in milliseconds

        Returns:
            The created AuditLogEntry
        """
        entry = AuditLogEntry(
            trace_id=trace_id or str(uuid4())[:12],
            session_id=session_id,
            user_id=user_id,
            action=action,
            resource=resource,
            details=details or {},
            ip_address=ip_address,
            duration_ms=duration_ms,
            log_level=level.upper()
        )

        # Log to file
        log_level = self._get_log_level(level)
        self.logger.log(log_level, entry.model_dump_json())

        # Add to buffer for queries
        async with self._buffer_lock:
            self._buffer.append(entry)
            # Keep buffer size manageable
            if len(self._buffer) > 10000:
                self._buffer = self._buffer[-5000:]

        # Periodic cleanup
        if time.time() - self._last_cleanup > 3600:  # Every hour
            asyncio.create_task(self._cleanup())

        return entry

    async def log_security(
        self,
        action: str,
        resource: str,
        details: Dict[str, Any] = None,
        **kwargs
    ) -> AuditLogEntry:
        """Convenience method for security events."""
        return await self.log("SECURITY", action, resource, details, **kwargs)

    async def query(
        self,
        limit: int = 100,
        level: Optional[str] = None,
        action: Optional[str] = None,
        trace_id: Optional[str] = None,
        since: Optional[datetime] = None
    ) -> List[Dict[str, Any]]:
        """
        Query recent audit entries from buffer.

        For historical queries, use export_for_siem and query the files.
        """
        async with self._buffer_lock:
            results = []
            for entry in reversed(self._buffer):
                if len(results) >= limit:
                    break

                # Apply filters
                if level and entry.log_level != level.upper():
                    continue
                if action and entry.action != action:
                    continue
                if trace_id and entry.trace_id != trace_id:
                    continue
                if since:
                    entry_time = datetime.fromisoformat(entry.timestamp.replace("Z", "+00:00"))
                    if entry_time < since:
                        continue

                results.append(entry.model_dump())

            return results

    async def export_for_siem(
        self,
        output_path: str,
        compress: bool = True,
        since: Optional[datetime] = None
    ) -> str:
        """
        Export audit logs for SIEM ingestion.

        Args:
            output_path: Output file path
            compress: Whether to gzip the output
            since: Only export logs since this datetime

        Returns:
            Path to the exported file
        """
        output_file = Path(output_path)
        output_file.parent.mkdir(parents=True, exist_ok=True)

        temp_file = output_file.with_suffix('.tmp')

        with temp_file.open("w", encoding="utf-8") as f:
            # Export from all log files
            for log_file in sorted(self.log_dir.glob("audit.jsonl*")):
                if log_file.suffix == '.gz':
                    continue  # Skip already compressed

                with log_file.open("r", encoding="utf-8") as log:
                    for line in log:
                        if since:
                            try:
                                entry = json.loads(line)
                                entry_time = datetime.fromisoformat(
                                    entry.get("timestamp", "").replace("Z", "+00:00")
                                )
                                if entry_time < since:
                                    continue
                            except (json.JSONDecodeError, ValueError):
                                continue
                        f.write(line)
                        if not line.endswith('\n'):
                            f.write('\n')

        if compress:
            final_path = output_file.with_suffix('.jsonl.gz')
            with temp_file.open('rb') as f_in:
                with gzip.open(final_path, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)
            temp_file.unlink()
            return str(final_path)
        else:
            temp_file.rename(output_file)
            return str(output_file)

    async def _cleanup(self):
        """Enforce retention policy."""
        self._last_cleanup = time.time()

        now = datetime.now()
        cutoff = now - timedelta(days=self.retention_days)

        # Remove old files
        for log_file in self.log_dir.glob("audit.jsonl.*"):
            try:
                mtime = datetime.fromtimestamp(log_file.stat().st_mtime)
                if mtime < cutoff:
                    # Compress before deletion if not already
                    if not log_file.suffix == '.gz':
                        archive_path = self.log_dir / "archive" / f"{log_file.name}.gz"
                        archive_path.parent.mkdir(exist_ok=True)
                        with log_file.open('rb') as f_in:
                            with gzip.open(archive_path, 'wb') as f_out:
                                shutil.copyfileobj(f_in, f_out)
                    log_file.unlink()
            except Exception:
                pass

        # Check total size
        total_size = sum(f.stat().st_size for f in self.log_dir.glob("*.jsonl*"))
        while total_size > self.max_bytes:
            # Remove oldest
            files = sorted(self.log_dir.glob("audit.jsonl.*"), key=lambda f: f.stat().st_mtime)
            if files:
                files[0].unlink()
                total_size = sum(f.stat().st_size for f in self.log_dir.glob("*.jsonl*"))
            else:
                break

    async def get_stats(self) -> Dict[str, Any]:
        """Get audit log statistics."""
        async with self._buffer_lock:
            level_counts = {}
            action_counts = {}

            for entry in self._buffer:
                level_counts[entry.log_level] = level_counts.get(entry.log_level, 0) + 1
                action_counts[entry.action] = action_counts.get(entry.action, 0) + 1

            return {
                "buffer_size": len(self._buffer),
                "by_level": level_counts,
                "by_action": dict(sorted(action_counts.items(), key=lambda x: -x[1])[:10]),
                "log_files": len(list(self.log_dir.glob("*.jsonl*"))),
                "total_size_mb": sum(f.stat().st_size for f in self.log_dir.glob("*.jsonl*")) / (1024 * 1024),
                "retention_days": self.retention_days
            }


# Global instance
enhanced_audit_logger = EnhancedAuditLogger()
