#!/usr/bin/env bash
# Ensure Bash even if invoked via sh

if [ -z "${BASH_VERSION:-}" ]; then

  exec /usr/bin/env bash "$0" "$@"

fi

#!/usr/bin/sh
set -euo pipefail

#
# generate-index.sh – Erstellt ein attraktives, statisches index.html
# Frontend für das Apt-Mirror-Verzeichnis.
#
# Steuerung via ENV:
#   HOST_REPO_PATH=/root/ailinux-repo/repo
#   BASE_URL=https://repo.ailinux.me:8443
#   PUBLIC_PATH=mirror
#

SCRIPT_DIR="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" &>/dev/null && pwd)"
HOST_REPO_PATH="${HOST_REPO_PATH:-$SCRIPT_DIR/repo}"
HOST_MIRROR_PATH="$HOST_REPO_PATH/mirror"
INDEX_FILE_ON_MIRROR="$HOST_MIRROR_PATH/index.html"

BASE_URL="${BASE_URL:-https://repo.ailinux.me:8443}"
BASE_URL="${BASE_URL%/}"
PUBLIC_PATH="${PUBLIC_PATH:-mirror}"
PUBLIC_PATH="${PUBLIC_PATH#/}"
PUBLIC_PATH="${PUBLIC_PATH%/}"
GPG_URL_REL="ailinux-archive-key.gpg"
ADD_REPO_URL_REL="add-ailinux-repo.sh"
LIVE_LOG_URL_REL="live-log.html"
SUMMARY_LOG_URL_REL="mirror-summary.html"

if [[ -n "$PUBLIC_PATH" ]]; then
  PUBLIC_BASE="$BASE_URL/$PUBLIC_PATH"
else
  PUBLIC_BASE="$BASE_URL"
fi

PUBLIC_BASE="${PUBLIC_BASE%/}"

GPG_URL_ABS="$PUBLIC_BASE/$GPG_URL_REL"
ADD_REPO_URL_ABS="$PUBLIC_BASE/$ADD_REPO_URL_REL"
LIVE_LOG_URL_ABS="$PUBLIC_BASE/$LIVE_LOG_URL_REL"
SUMMARY_LOG_URL_ABS="$PUBLIC_BASE/$SUMMARY_LOG_URL_REL"

mkdir -p "$HOST_MIRROR_PATH"

TMP_INDEX_FILE="$(mktemp)"
trap 'rm -f "$TMP_INDEX_FILE"' EXIT

cat <<EOF_INDEX > "$TMP_INDEX_FILE"
<!DOCTYPE html>
<html lang="de">
<head>
<meta charset="UTF-8">
<title>🧠 AILinux Mirror Directory</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
body { background: #101820; color: #00ffaa; font-family: 'Courier New', Courier, monospace; padding: 20px; line-height: 1.6; }
h1, h2 { color: #6cf; border-bottom: 1px solid #444;}
a { color: #00ffff; text-decoration: none; }
a:hover { text-decoration: underline; color: #ffffff; }
ul { padding-left: 0; }
li { margin: 5px 0; list-style: none; word-break: break-all; }
li.dir a::before { content: '📁 '; color: #aaa; }
li.file a::before { content: '📄 '; color: #aaa; }
li.log a::before { content: '📄 '; color: #aaa; }
li.gpg a::before { content: '🔐 '; color: #aaa; }
li.script a::before { content: '⚙️ '; color: #aaa; }
pre { background: #222; padding: 15px; border: 1px solid #444; border-radius: 5px; overflow-x: auto; color: #eee; }
code { font-family: 'Courier New', Courier, monospace; }
hr { border: 0; border-top: 1px solid #444; margin: 2em 0; }
footer { margin-top: 2em; font-size: 0.8em; color: #555; text-align: center; }
</style>
</head>
<body>
<h1>📦 AILinux Mirror Directory</h1>
<p><strong>Letzte Aktualisierung:</strong> $(date '+%Y-%m-%d %H:%M:%S')</p>

<h2>🔐 GPG Key Installation</h2>
<pre><code>sudo install -d /usr/share/keyrings
curl -fsSL "$GPG_URL_ABS" | sudo gpg --dearmor --yes --output /usr/share/keyrings/ailinux-archive-keyring.gpg</code></pre>
<p><span class="gpg"><a href="$GPG_URL_ABS">$GPG_URL_REL</a></span> direkt herunterladen.</p>

<h2>⚙ Automatische Repo-Einbindung</h2>
<pre><code>curl -fsSL "$ADD_REPO_URL_ABS" | sudo bash</code></pre>
<p><span class="script"><a href="$ADD_REPO_URL_ABS">$ADD_REPO_URL_REL</a></span> Skript ansehen/herunterladen.</p>

<hr>
<h2>📄 Logs & Info</h2>
<ul>
    <li class="log"><a href="$LIVE_LOG_URL_ABS">$LIVE_LOG_URL_REL</a> (Aktueller Mirror-Prozess)</li>
    <li class="log"><a href="$SUMMARY_LOG_URL_ABS">$SUMMARY_LOG_URL_REL</a> (Letzte Post-Mirror Zusammenfassung)</li>
</ul>

<hr>
<h2>📂 Verzeichnisübersicht</h2>
<ul>
<li class="gpg"><a href="$GPG_URL_ABS">$GPG_URL_REL</a></li>
<li class="script"><a href="$ADD_REPO_URL_ABS">$ADD_REPO_URL_REL</a></li>
<hr style="border-top: 1px dashed #444; margin: 0.5em 0;">
EOF_INDEX

find "$HOST_MIRROR_PATH" -maxdepth 1 -mindepth 1 \
  -not -name "index.html" \
  -not -name "live-log.html" \
  -not -name "mirror-summary.html" \
  -not -name "ailinux-archive-key.gpg" \
  -not -name "ailinux.gpg" \
  -not -name "add-ailinux-repo.sh" | sort | while read -r path; do
  name=$(basename "$path")
  if [ -d "$path" ]; then
    echo "<li class=\"dir\"><a href=\"$PUBLIC_BASE/$name/\">$name/</a></li>" >> "$TMP_INDEX_FILE"
  elif [ -f "$path" ]; then
    echo "<li class=\"file\"><a href=\"$PUBLIC_BASE/$name\">$name</a></li>" >> "$TMP_INDEX_FILE"
  fi
done

cat <<EOF_FOOTER >> "$TMP_INDEX_FILE"
</ul>

<hr>
<footer>
AILinux Repository © $(date +%Y) – derleiti.de<br>
Index generated by 🧠 Nova AI on $(date '+%F %T')
</footer>
</body>
</html>
EOF_FOOTER

declare -a INDEX_TARGETS
INDEX_TARGETS+=("$INDEX_FILE_ON_MIRROR")

if [[ "${WRITE_ROOT_INDEX:-1}" == "1" ]]; then
  INDEX_TARGETS+=("$HOST_REPO_PATH/index.html")
fi

if [[ -n "${EXTRA_INDEX_TARGETS:-}" ]]; then
  while IFS= read -r extra_target; do
    [[ -n "$extra_target" ]] || continue
    INDEX_TARGETS+=("$extra_target")
  done <<< "${EXTRA_INDEX_TARGETS}"
fi

declare -A GENERATED
for target in "${INDEX_TARGETS[@]}"; do
  [[ -n "$target" ]] || continue
  if [[ -n "${GENERATED[$target]+x}" ]]; then
    continue
  fi
  GENERATED["$target"]=1
  mkdir -p "$(dirname "$target")"
  cp "$TMP_INDEX_FILE" "$target"
  chmod 644 "$target"
  echo "[generate-index] $target erfolgreich erstellt."
done


