"""
Prometheus Metrics for AILinux Backend
Generated by Codestral, reviewed by Claude

Provides metrics for:
- Request count and latency
- LLM calls and latency
- Active connections
- Memory entries
- Circuit breaker states
"""

from prometheus_client import Counter, Histogram, Gauge, generate_latest, CONTENT_TYPE_LATEST
from fastapi import Request, Response
import time
import logging

logger = logging.getLogger("ailinux.metrics")

# Request Metrics
REQUEST_COUNT = Counter(
    'ailinux_request_count',
    'Total HTTP requests',
    ['endpoint', 'method', 'status']
)

REQUEST_LATENCY = Histogram(
    'ailinux_request_latency_seconds',
    'Request latency in seconds',
    ['endpoint'],
    buckets=(0.01, 0.025, 0.05, 0.1, 0.25, 0.5, 1.0, 2.5, 5.0, 10.0)
)

# LLM Metrics
LLM_CALLS_TOTAL = Counter(
    'ailinux_llm_calls_total',
    'Total LLM API calls',
    ['model', 'provider', 'status']
)

LLM_LATENCY_SECONDS = Histogram(
    'ailinux_llm_latency_seconds',
    'LLM call latency in seconds',
    ['model', 'provider'],
    buckets=(0.1, 0.5, 1.0, 2.5, 5.0, 10.0, 30.0, 60.0, 120.0)
)

LLM_TOKENS_TOTAL = Counter(
    'ailinux_llm_tokens_total',
    'Total tokens processed',
    ['model', 'provider', 'type']  # type: input, output
)

# Connection Metrics
ACTIVE_CONNECTIONS = Gauge(
    'ailinux_active_connections',
    'Number of active connections'
)

WEBSOCKET_CONNECTIONS = Gauge(
    'ailinux_websocket_connections',
    'Number of active WebSocket connections'
)

# Memory Metrics
MEMORY_ENTRIES_TOTAL = Gauge(
    'ailinux_memory_entries_total',
    'Total memory entries in TriForce'
)

MEMORY_ENTRIES_BY_TYPE = Gauge(
    'ailinux_memory_entries_by_type',
    'Memory entries by type',
    ['type']
)

# Circuit Breaker Metrics
CIRCUIT_BREAKER_STATE = Gauge(
    'ailinux_circuit_breaker_state',
    'Circuit breaker state (0=closed, 1=open, 2=half_open)',
    ['llm_id']
)

CIRCUIT_BREAKER_FAILURES = Counter(
    'ailinux_circuit_breaker_failures_total',
    'Total circuit breaker failures',
    ['llm_id']
)

# Queue Metrics
QUEUE_SIZE = Gauge(
    'ailinux_queue_size',
    'Current queue size',
    ['queue_name']
)

QUEUE_PROCESSING_TIME = Histogram(
    'ailinux_queue_processing_seconds',
    'Queue item processing time',
    ['queue_name']
)


class MetricsMiddleware:
    """Middleware to collect request metrics."""

    async def __call__(self, request: Request, call_next):
        start_time = time.time()
        ACTIVE_CONNECTIONS.inc()

        try:
            response = await call_next(request)
            process_time = time.time() - start_time

            # Skip metrics endpoint to avoid recursion
            if request.url.path != "/metrics":
                endpoint = self._normalize_endpoint(request.url.path)
                method = request.method
                status = str(response.status_code)

                REQUEST_COUNT.labels(
                    endpoint=endpoint,
                    method=method,
                    status=status
                ).inc()

                REQUEST_LATENCY.labels(endpoint=endpoint).observe(process_time)

            return response
        finally:
            ACTIVE_CONNECTIONS.dec()

    def _normalize_endpoint(self, path: str) -> str:
        """Normalize endpoint path to reduce cardinality."""
        # Replace IDs with placeholders
        parts = path.split('/')
        normalized = []
        for part in parts:
            if part and (part.isdigit() or len(part) == 36 or part.startswith('mem_')):
                normalized.append('{id}')
            else:
                normalized.append(part)
        return '/'.join(normalized)


def get_metrics_response() -> Response:
    """Generate Prometheus metrics response."""
    return Response(
        generate_latest(),
        media_type=CONTENT_TYPE_LATEST
    )


# Helper functions for recording metrics
def record_llm_call(model: str, provider: str, status: str = "success"):
    """Record an LLM call."""
    LLM_CALLS_TOTAL.labels(model=model, provider=provider, status=status).inc()


def record_llm_latency(model: str, provider: str, duration: float):
    """Record LLM call latency."""
    LLM_LATENCY_SECONDS.labels(model=model, provider=provider).observe(duration)


def record_llm_tokens(model: str, provider: str, input_tokens: int, output_tokens: int):
    """Record token usage."""
    LLM_TOKENS_TOTAL.labels(model=model, provider=provider, type="input").inc(input_tokens)
    LLM_TOKENS_TOTAL.labels(model=model, provider=provider, type="output").inc(output_tokens)


def set_circuit_breaker_state(llm_id: str, state: int):
    """Set circuit breaker state (0=closed, 1=open, 2=half_open)."""
    CIRCUIT_BREAKER_STATE.labels(llm_id=llm_id).set(state)


def record_circuit_breaker_failure(llm_id: str):
    """Record a circuit breaker failure."""
    CIRCUIT_BREAKER_FAILURES.labels(llm_id=llm_id).inc()


def set_memory_entries(total: int, by_type: dict = None):
    """Set memory entry counts."""
    MEMORY_ENTRIES_TOTAL.set(total)
    if by_type:
        for mem_type, count in by_type.items():
            MEMORY_ENTRIES_BY_TYPE.labels(type=mem_type).set(count)


def set_queue_size(queue_name: str, size: int):
    """Set queue size."""
    QUEUE_SIZE.labels(queue_name=queue_name).set(size)


def record_queue_processing(queue_name: str, duration: float):
    """Record queue processing time."""
    QUEUE_PROCESSING_TIME.labels(queue_name=queue_name).observe(duration)
