/**
 * Use this to connect authentication providers that support OAuth 2.0.
 *
 * ```ts {5-12}
 * import { Oauth2Provider } from "@openauthjs/openauth/provider/oauth2"
 *
 * export default issuer({
 *   providers: {
 *     oauth2: Oauth2Provider({
 *       clientID: "1234567890",
 *       clientSecret: "0987654321",
 *       endpoint: {
 *         authorization: "https://auth.myserver.com/authorize",
 *         token: "https://auth.myserver.com/token"
 *       }
 *     })
 *   }
 * })
 * ```
 *
 *
 * @packageDocumentation
 */
import { Provider } from "./provider.js";
export interface Oauth2Config {
    /**
     * @internal
     */
    type?: string;
    /**
     * The client ID.
     *
     * This is just a string to identify your app.
     *
     * @example
     * ```ts
     * {
     *   clientID: "my-client"
     * }
     * ```
     */
    clientID: string;
    /**
     * The client secret.
     *
     * This is a private key that's used to authenticate your app. It should be kept secret.
     *
     * @example
     * ```ts
     * {
     *   clientSecret: "0987654321"
     * }
     * ```
     */
    clientSecret: string;
    /**
     * The URLs of the authorization and token endpoints.
     *
     * @example
     * ```ts
     * {
     *   endpoint: {
     *     authorization: "https://auth.myserver.com/authorize",
     *     token: "https://auth.myserver.com/token"
     *   }
     * }
     * ```
     */
    endpoint: {
        /**
         * The URL of the authorization endpoint.
         */
        authorization: string;
        /**
         * The URL of the token endpoint.
         */
        token: string;
    };
    /**
     * A list of OAuth scopes that you want to request.
     *
     * @example
     * ```ts
     * {
     *   scopes: ["email", "profile"]
     * }
     * ```
     */
    scopes: string[];
    /**
     * Whether to use PKCE (Proof Key for Code Exchange) for the authorization code flow.
     * Some providers like x.com require this.
     * @default false
     */
    pkce?: boolean;
    /**
     * Any additional parameters that you want to pass to the authorization endpoint.
     * @example
     * ```ts
     * {
     *   query: {
     *     access_type: "offline",
     *     prompt: "consent"
     *   }
     * }
     * ```
     */
    query?: Record<string, string>;
}
/**
 * @internal
 */
export type Oauth2WrappedConfig = Omit<Oauth2Config, "endpoint" | "name">;
/**
 * @internal
 */
export interface Oauth2Token {
    access: string;
    refresh: string;
    expiry: number;
    raw: Record<string, any>;
}
export declare function Oauth2Provider(config: Oauth2Config): Provider<{
    tokenset: Oauth2Token;
    clientID: string;
}>;
//# sourceMappingURL=oauth2.d.ts.map