// src/provider/password.ts
import { UnknownStateError } from "../error.js";
import { Storage } from "../storage/storage.js";
import { generateUnbiasedDigits, timingSafeCompare } from "../random.js";
import * as jose from "jose";
import { TextEncoder } from "node:util";
import { timingSafeEqual, randomBytes, scrypt } from "node:crypto";
import { getRelativeUrl } from "../util.js";
function PasswordProvider(config) {
  const hasher = config.hasher ?? ScryptHasher();
  function generate() {
    return generateUnbiasedDigits(6);
  }
  return {
    type: "password",
    init(routes, ctx) {
      routes.get("/authorize", async (c) => ctx.forward(c, await config.login(c.req.raw)));
      routes.post("/authorize", async (c) => {
        const fd = await c.req.formData();
        async function error(err) {
          return ctx.forward(c, await config.login(c.req.raw, fd, err));
        }
        const email = fd.get("email")?.toString()?.toLowerCase();
        if (!email)
          return error({ type: "invalid_email" });
        const hash = await Storage.get(ctx.storage, [
          "email",
          email,
          "password"
        ]);
        const password = fd.get("password")?.toString();
        if (!password || !hash || !await hasher.verify(password, hash))
          return error({ type: "invalid_password" });
        return ctx.success(c, {
          email
        }, {
          invalidate: async (subject) => {
            await Storage.set(ctx.storage, ["email", email, "subject"], subject);
          }
        });
      });
      routes.get("/register", async (c) => {
        const state = {
          type: "start"
        };
        await ctx.set(c, "provider", 60 * 60 * 24, state);
        return ctx.forward(c, await config.register(c.req.raw, state));
      });
      routes.post("/register", async (c) => {
        const fd = await c.req.formData();
        const email = fd.get("email")?.toString()?.toLowerCase();
        const action = fd.get("action")?.toString();
        const provider = await ctx.get(c, "provider");
        async function transition(next, err) {
          await ctx.set(c, "provider", 60 * 60 * 24, next);
          return ctx.forward(c, await config.register(c.req.raw, next, fd, err));
        }
        if (action === "register" && provider.type === "start") {
          const password = fd.get("password")?.toString();
          const repeat = fd.get("repeat")?.toString();
          if (!email)
            return transition(provider, { type: "invalid_email" });
          if (!password)
            return transition(provider, { type: "invalid_password" });
          if (password !== repeat)
            return transition(provider, { type: "password_mismatch" });
          if (config.validatePassword) {
            let validationError;
            try {
              if (typeof config.validatePassword === "function") {
                validationError = await config.validatePassword(password);
              } else {
                const res = await config.validatePassword["~standard"].validate(password);
                if (res.issues?.length) {
                  throw new Error(res.issues.map((issue) => issue.message).join(", "));
                }
              }
            } catch (error) {
              validationError = error instanceof Error ? error.message : undefined;
            }
            if (validationError)
              return transition(provider, {
                type: "validation_error",
                message: validationError
              });
          }
          const existing = await Storage.get(ctx.storage, [
            "email",
            email,
            "password"
          ]);
          if (existing)
            return transition(provider, { type: "email_taken" });
          const code = generate();
          await config.sendCode(email, code);
          return transition({
            type: "code",
            code,
            password: await hasher.hash(password),
            email
          });
        }
        if (action === "register" && provider.type === "code") {
          const code = generate();
          await config.sendCode(provider.email, code);
          return transition({
            type: "code",
            code,
            password: provider.password,
            email: provider.email
          });
        }
        if (action === "verify" && provider.type === "code") {
          const code = fd.get("code")?.toString();
          if (!code || !timingSafeCompare(code, provider.code))
            return transition(provider, { type: "invalid_code" });
          const existing = await Storage.get(ctx.storage, [
            "email",
            provider.email,
            "password"
          ]);
          if (existing)
            return transition({ type: "start" }, { type: "email_taken" });
          await Storage.set(ctx.storage, ["email", provider.email, "password"], provider.password);
          return ctx.success(c, {
            email: provider.email
          });
        }
        return transition({ type: "start" });
      });
      routes.get("/change", async (c) => {
        let redirect = c.req.query("redirect_uri") || getRelativeUrl(c, "./authorize");
        const state = {
          type: "start",
          redirect
        };
        await ctx.set(c, "provider", 60 * 60 * 24, state);
        return ctx.forward(c, await config.change(c.req.raw, state));
      });
      routes.post("/change", async (c) => {
        const fd = await c.req.formData();
        const action = fd.get("action")?.toString();
        const provider = await ctx.get(c, "provider");
        if (!provider)
          throw new UnknownStateError;
        async function transition(next, err) {
          await ctx.set(c, "provider", 60 * 60 * 24, next);
          return ctx.forward(c, await config.change(c.req.raw, next, fd, err));
        }
        if (action === "code") {
          const email = fd.get("email")?.toString()?.toLowerCase();
          if (!email)
            return transition({ type: "start", redirect: provider.redirect }, { type: "invalid_email" });
          const code = generate();
          await config.sendCode(email, code);
          return transition({
            type: "code",
            code,
            email,
            redirect: provider.redirect
          });
        }
        if (action === "verify" && provider.type === "code") {
          const code = fd.get("code")?.toString();
          if (!code || !timingSafeCompare(code, provider.code))
            return transition(provider, { type: "invalid_code" });
          return transition({
            type: "update",
            email: provider.email,
            redirect: provider.redirect
          });
        }
        if (action === "update" && provider.type === "update") {
          const existing = await Storage.get(ctx.storage, [
            "email",
            provider.email,
            "password"
          ]);
          if (!existing)
            return c.redirect(provider.redirect, 302);
          const password = fd.get("password")?.toString();
          const repeat = fd.get("repeat")?.toString();
          if (!password)
            return transition(provider, { type: "invalid_password" });
          if (password !== repeat)
            return transition(provider, { type: "password_mismatch" });
          if (config.validatePassword) {
            let validationError;
            try {
              if (typeof config.validatePassword === "function") {
                validationError = await config.validatePassword(password);
              } else {
                const res = await config.validatePassword["~standard"].validate(password);
                if (res.issues?.length) {
                  throw new Error(res.issues.map((issue) => issue.message).join(", "));
                }
              }
            } catch (error) {
              validationError = error instanceof Error ? error.message : undefined;
            }
            if (validationError)
              return transition(provider, {
                type: "validation_error",
                message: validationError
              });
          }
          await Storage.set(ctx.storage, ["email", provider.email, "password"], await hasher.hash(password));
          const subject = await Storage.get(ctx.storage, [
            "email",
            provider.email,
            "subject"
          ]);
          if (subject)
            await ctx.invalidate(subject);
          return c.redirect(provider.redirect, 302);
        }
        return transition({ type: "start", redirect: provider.redirect });
      });
    }
  };
}
function PBKDF2Hasher(opts) {
  const iterations = opts?.iterations ?? 600000;
  return {
    async hash(password) {
      const encoder = new TextEncoder;
      const bytes = encoder.encode(password);
      const salt = crypto.getRandomValues(new Uint8Array(16));
      const keyMaterial = await crypto.subtle.importKey("raw", bytes, "PBKDF2", false, ["deriveBits"]);
      const hash = await crypto.subtle.deriveBits({
        name: "PBKDF2",
        hash: "SHA-256",
        salt,
        iterations
      }, keyMaterial, 256);
      const hashBase64 = jose.base64url.encode(new Uint8Array(hash));
      const saltBase64 = jose.base64url.encode(salt);
      return {
        hash: hashBase64,
        salt: saltBase64,
        iterations
      };
    },
    async verify(password, compare) {
      const encoder = new TextEncoder;
      const passwordBytes = encoder.encode(password);
      const salt = jose.base64url.decode(compare.salt);
      const params = {
        name: "PBKDF2",
        hash: "SHA-256",
        salt,
        iterations: compare.iterations
      };
      const keyMaterial = await crypto.subtle.importKey("raw", passwordBytes, "PBKDF2", false, ["deriveBits"]);
      const hash = await crypto.subtle.deriveBits(params, keyMaterial, 256);
      const hashBase64 = jose.base64url.encode(new Uint8Array(hash));
      return hashBase64 === compare.hash;
    }
  };
}
function ScryptHasher(opts) {
  const N = opts?.N ?? 16384;
  const r = opts?.r ?? 8;
  const p = opts?.p ?? 1;
  return {
    async hash(password) {
      const salt = randomBytes(16);
      const keyLength = 32;
      const derivedKey = await new Promise((resolve, reject) => {
        scrypt(password, salt, keyLength, { N, r, p }, (err, derivedKey2) => {
          if (err)
            reject(err);
          else
            resolve(derivedKey2);
        });
      });
      const hashBase64 = derivedKey.toString("base64");
      const saltBase64 = salt.toString("base64");
      return {
        hash: hashBase64,
        salt: saltBase64,
        N,
        r,
        p
      };
    },
    async verify(password, compare) {
      const salt = Buffer.from(compare.salt, "base64");
      const keyLength = 32;
      const derivedKey = await new Promise((resolve, reject) => {
        scrypt(password, salt, keyLength, { N: compare.N, r: compare.r, p: compare.p }, (err, derivedKey2) => {
          if (err)
            reject(err);
          else
            resolve(derivedKey2);
        });
      });
      return timingSafeEqual(derivedKey, Buffer.from(compare.hash, "base64"));
    }
  };
}
export {
  ScryptHasher,
  PasswordProvider,
  PBKDF2Hasher
};
